#!/bin/bash
# backup-current-config.sh
# SMART BACKUP: Captures YOUR current settings, not system defaults
# This preserves your existing kernel tuning when you restore

set -e

TIMESTAMP=$(date +%Y%m%d-%H%M%S)
BACKUP_DIR="/root/racing-tuning-backup-${TIMESTAMP}"

echo "==================================="
echo "Smart Configuration Backup"
echo "==================================="
echo
echo "Creating backup of CURRENT settings..."
echo "Backup location: $BACKUP_DIR"
echo

mkdir -p "$BACKUP_DIR"

# ============================================================================
# DETECT SWIZZIN
# ============================================================================

SWIZZIN_USER=""
if [ -f "/install/.swizzin.lock" ] || [ -f "/etc/swizzin/sources/globals" ]; then
    SWIZZIN_USER=$(ls /home | grep -v "lost+found" | head -1)
    echo "✓ Swizzin detected, user: $SWIZZIN_USER"
fi

# ============================================================================
# BACKUP SYSCTL SETTINGS (CURRENT VALUES)
# ============================================================================

echo
echo "Backing up current sysctl settings..."

# Backup entire sysctl state (all current values)
sysctl -a > "$BACKUP_DIR/sysctl-all-current.txt" 2>/dev/null
echo "✓ Full sysctl dump saved"

# Backup specific racing-related parameters with current values
cat > "$BACKUP_DIR/sysctl-restore.conf" <<EOF
# CURRENT SYSCTL VALUES (before racing tuning)
# Generated: $(date)
# These are YOUR tuned values, not system defaults

# TCP Buffers (current)
net.core.rmem_max = $(sysctl -n net.core.rmem_max)
net.core.wmem_max = $(sysctl -n net.core.wmem_max)
net.core.rmem_default = $(sysctl -n net.core.rmem_default)
net.core.wmem_default = $(sysctl -n net.core.wmem_default)
net.ipv4.tcp_rmem = $(sysctl -n net.ipv4.tcp_rmem)
net.ipv4.tcp_wmem = $(sysctl -n net.ipv4.tcp_wmem)
net.ipv4.tcp_mem = $(sysctl -n net.ipv4.tcp_mem)

# UDP Buffers (current)
net.ipv4.udp_rmem_min = $(sysctl -n net.ipv4.udp_rmem_min 2>/dev/null || echo "4096")
net.ipv4.udp_wmem_min = $(sysctl -n net.ipv4.udp_wmem_min 2>/dev/null || echo "4096")

# Congestion Control (current)
net.core.default_qdisc = $(sysctl -n net.core.default_qdisc)
net.ipv4.tcp_congestion_control = $(sysctl -n net.ipv4.tcp_congestion_control)

# TCP Options (current)
net.ipv4.tcp_slow_start_after_idle = $(sysctl -n net.ipv4.tcp_slow_start_after_idle)
net.ipv4.tcp_no_metrics_save = $(sysctl -n net.ipv4.tcp_no_metrics_save 2>/dev/null || echo "0")
net.ipv4.tcp_window_scaling = $(sysctl -n net.ipv4.tcp_window_scaling)
net.ipv4.tcp_timestamps = $(sysctl -n net.ipv4.tcp_timestamps)
net.ipv4.tcp_sack = $(sysctl -n net.ipv4.tcp_sack)
net.ipv4.tcp_autocorking = $(sysctl -n net.ipv4.tcp_autocorking)
net.ipv4.tcp_notsent_lowat = $(sysctl -n net.ipv4.tcp_notsent_lowat)

# TCP Fast Open (current)
net.ipv4.tcp_fastopen = $(sysctl -n net.ipv4.tcp_fastopen)
net.ipv4.tcp_fastopen_blackhole_timeout_sec = $(sysctl -n net.ipv4.tcp_fastopen_blackhole_timeout_sec 2>/dev/null || echo "3600")

# TCP Timeouts (current)
net.ipv4.tcp_fin_timeout = $(sysctl -n net.ipv4.tcp_fin_timeout)
net.ipv4.tcp_tw_reuse = $(sysctl -n net.ipv4.tcp_tw_reuse)
net.ipv4.tcp_keepalive_time = $(sysctl -n net.ipv4.tcp_keepalive_time)
net.ipv4.tcp_keepalive_probes = $(sysctl -n net.ipv4.tcp_keepalive_probes)
net.ipv4.tcp_keepalive_intvl = $(sysctl -n net.ipv4.tcp_keepalive_intvl)
net.ipv4.tcp_syn_retries = $(sysctl -n net.ipv4.tcp_syn_retries)
net.ipv4.tcp_synack_retries = $(sysctl -n net.ipv4.tcp_synack_retries)

# TCP Socket Limits (current)
net.ipv4.tcp_max_orphans = $(sysctl -n net.ipv4.tcp_max_orphans)
net.ipv4.tcp_orphan_retries = $(sysctl -n net.ipv4.tcp_orphan_retries)

# Queue Tuning (current)
net.core.netdev_max_backlog = $(sysctl -n net.core.netdev_max_backlog)
net.ipv4.tcp_max_syn_backlog = $(sysctl -n net.ipv4.tcp_max_syn_backlog)
net.core.somaxconn = $(sysctl -n net.core.somaxconn)

# Busy Polling (current)
net.core.busy_read = $(sysctl -n net.core.busy_read)
net.core.busy_poll = $(sysctl -n net.core.busy_poll)

# Packet Scheduling (current)
net.ipv4.tcp_limit_output_bytes = $(sysctl -n net.ipv4.tcp_limit_output_bytes 2>/dev/null || echo "262144")
net.ipv4.tcp_pacing_ss_ratio = $(sysctl -n net.ipv4.tcp_pacing_ss_ratio 2>/dev/null || echo "200")
net.ipv4.tcp_pacing_ca_ratio = $(sysctl -n net.ipv4.tcp_pacing_ca_ratio 2>/dev/null || echo "120")

# File System (current)
fs.file-max = $(sysctl -n fs.file-max)
fs.aio-max-nr = $(sysctl -n fs.aio-max-nr)
fs.inotify.max_user_watches = $(sysctl -n fs.inotify.max_user_watches)
fs.inotify.max_user_instances = $(sysctl -n fs.inotify.max_user_instances)

# VM Tuning (current)
vm.dirty_ratio = $(sysctl -n vm.dirty_ratio)
vm.dirty_background_ratio = $(sysctl -n vm.dirty_background_ratio)
vm.dirty_expire_centisecs = $(sysctl -n vm.dirty_expire_centisecs)
vm.dirty_writeback_centisecs = $(sysctl -n vm.dirty_writeback_centisecs)
vm.vfs_cache_pressure = $(sysctl -n vm.vfs_cache_pressure)
vm.swappiness = $(sysctl -n vm.swappiness)

# CPU Scheduler (current)
kernel.sched_wakeup_granularity_ns = $(sysctl -n kernel.sched_wakeup_granularity_ns 2>/dev/null || echo "1000000")
kernel.sched_min_granularity_ns = $(sysctl -n kernel.sched_min_granularity_ns 2>/dev/null || echo "10000000")
kernel.sched_migration_cost_ns = $(sysctl -n kernel.sched_migration_cost_ns 2>/dev/null || echo "500000")
kernel.sched_latency_ns = $(sysctl -n kernel.sched_latency_ns 2>/dev/null || echo "24000000")
kernel.sched_nr_migrate = $(sysctl -n kernel.sched_nr_migrate 2>/dev/null || echo "32")
kernel.sched_rt_runtime_us = $(sysctl -n kernel.sched_rt_runtime_us 2>/dev/null || echo "950000")
kernel.sched_rt_period_us = $(sysctl -n kernel.sched_rt_period_us 2>/dev/null || echo "1000000")

# Network Device (current)
net.core.netdev_budget = $(sysctl -n net.core.netdev_budget)
net.core.netdev_budget_usecs = $(sysctl -n net.core.netdev_budget_usecs)
net.core.rps_sock_flow_entries = $(sysctl -n net.core.rps_sock_flow_entries 2>/dev/null || echo "0")

# ARP Cache (current)
net.ipv4.neigh.default.gc_thresh1 = $(sysctl -n net.ipv4.neigh.default.gc_thresh1)
net.ipv4.neigh.default.gc_thresh2 = $(sysctl -n net.ipv4.neigh.default.gc_thresh2)
net.ipv4.neigh.default.gc_thresh3 = $(sysctl -n net.ipv4.neigh.default.gc_thresh3)

# IP Forwarding (current)
net.ipv4.ip_forward = $(sysctl -n net.ipv4.ip_forward)

# Security Settings (current)
net.ipv4.conf.all.rp_filter = $(sysctl -n net.ipv4.conf.all.rp_filter)
net.ipv4.conf.default.rp_filter = $(sysctl -n net.ipv4.conf.default.rp_filter)
net.ipv4.icmp_ratelimit = $(sysctl -n net.ipv4.icmp_ratelimit)

# Additional Options (current)
net.ipv4.tcp_ecn = $(sysctl -n net.ipv4.tcp_ecn)
net.ipv4.tcp_rto_min = $(sysctl -n net.ipv4.tcp_rto_min 2>/dev/null || echo "200")
net.ipv4.tcp_mtu_probing = $(sysctl -n net.ipv4.tcp_mtu_probing)
net.ipv4.tcp_base_mss = $(sysctl -n net.ipv4.tcp_base_mss)

# Connection Tracking (current - may not exist)
net.nf_conntrack_max = $(sysctl -n net.nf_conntrack_max 2>/dev/null || echo "# Not available")
EOF

echo "✓ Sysctl restore config created"

# ============================================================================
# BACKUP EXISTING SYSCTL FILES
# ============================================================================

if [ -f /etc/sysctl.conf ]; then
    cp /etc/sysctl.conf "$BACKUP_DIR/sysctl.conf.original"
    echo "✓ Backed up /etc/sysctl.conf"
fi

if [ -d /etc/sysctl.d ]; then
    mkdir -p "$BACKUP_DIR/sysctl.d"
    cp -r /etc/sysctl.d/* "$BACKUP_DIR/sysctl.d/" 2>/dev/null || true
    echo "✓ Backed up /etc/sysctl.d/"
fi

# ============================================================================
# BACKUP QBITTORRENT CONFIGURATION
# ============================================================================

if [ -n "$SWIZZIN_USER" ]; then
    QBIT_CONFIG="/home/$SWIZZIN_USER/.config/qBittorrent/qBittorrent.conf"
    if [ -f "$QBIT_CONFIG" ]; then
        cp "$QBIT_CONFIG" "$BACKUP_DIR/qBittorrent.conf.original"
        echo "✓ Backed up qBittorrent configuration"
    fi
fi

# ============================================================================
# BACKUP NETWORK SETTINGS
# ============================================================================

echo
echo "Backing up network settings..."

# Find active network interface
IFACE=$(ip route | grep default | awk '{print $5}' | head -1)

if [ -n "$IFACE" ]; then
    cat > "$BACKUP_DIR/network-settings.txt" <<EOF
# Network Interface Settings (before racing tuning)
# Interface: $IFACE
# Generated: $(date)

# TX Queue Length
TX_QLEN=$(cat /sys/class/net/$IFACE/tx_queue_len 2>/dev/null || echo "unknown")

# Interrupt Coalescing (ethtool -c)
$(ethtool -c $IFACE 2>/dev/null || echo "ethtool coalescing not available")

# Offload Settings (ethtool -k)
$(ethtool -k $IFACE 2>/dev/null || echo "ethtool offload settings not available")

# Ring Buffer Settings (ethtool -g)
$(ethtool -g $IFACE 2>/dev/null || echo "ethtool ring settings not available")

# Pause Frames (ethtool -a)
$(ethtool -a $IFACE 2>/dev/null || echo "ethtool pause settings not available")
EOF
    echo "✓ Network settings backed up for $IFACE"
fi

# ============================================================================
# BACKUP EXISTING TUNING SCRIPTS
# ============================================================================

if [ -f /usr/local/bin/tune-hardware.sh ]; then
    cp /usr/local/bin/tune-hardware.sh "$BACKUP_DIR/tune-hardware.sh.old"
    echo "✓ Backed up existing tune-hardware.sh"
fi

if [ -f /usr/local/bin/tune-cpu-io.sh ]; then
    cp /usr/local/bin/tune-cpu-io.sh "$BACKUP_DIR/tune-cpu-io.sh.old"
    echo "✓ Backed up existing tune-cpu-io.sh"
fi

# ============================================================================
# BACKUP SYSTEMD SERVICES
# ============================================================================

if [ -f /etc/systemd/system/seedbox-tuning.service ]; then
    cp /etc/systemd/system/seedbox-tuning.service "$BACKUP_DIR/"
    echo "✓ Backed up seedbox-tuning.service"
fi

if [ -n "$SWIZZIN_USER" ]; then
    if [ -f /etc/systemd/system/qbittorrent@.service.d/racing-override.conf ]; then
        mkdir -p "$BACKUP_DIR/systemd"
        cp /etc/systemd/system/qbittorrent@.service.d/racing-override.conf "$BACKUP_DIR/systemd/"
        echo "✓ Backed up qBittorrent RT override"
    fi
fi

# ============================================================================
# BACKUP LIMITS CONFIGURATION
# ============================================================================

if [ -f /etc/security/limits.conf ]; then
    cp /etc/security/limits.conf "$BACKUP_DIR/limits.conf.original"
    echo "✓ Backed up /etc/security/limits.conf"
fi

# ============================================================================
# CREATE RESTORE SCRIPT
# ============================================================================

cat > "$BACKUP_DIR/RESTORE.sh" <<'EOFSCRIPT'
#!/bin/bash
# Auto-generated restore script
# Restores to backed-up configuration (your tuned state)

set -e

BACKUP_DIR="$(dirname "$0")"

echo "Restoring from: $BACKUP_DIR"
echo

# Restore sysctl settings
if [ -f "$BACKUP_DIR/sysctl-restore.conf" ]; then
    echo "Restoring sysctl settings..."
    sysctl -p "$BACKUP_DIR/sysctl-restore.conf"
    echo "✓ Restored"
fi

# Restore original sysctl.conf if it exists
if [ -f "$BACKUP_DIR/sysctl.conf.original" ]; then
    cp "$BACKUP_DIR/sysctl.conf.original" /etc/sysctl.conf
    echo "✓ Restored /etc/sysctl.conf"
fi

# Restore sysctl.d if it exists
if [ -d "$BACKUP_DIR/sysctl.d" ]; then
    # Remove racing config
    rm -f /etc/sysctl.d/99-seedbox-racing.conf
    echo "✓ Removed racing sysctl config"
fi

echo
echo "Restore complete. Reboot recommended."
EOFSCRIPT

chmod +x "$BACKUP_DIR/RESTORE.sh"
echo "✓ Created restore script"

# ============================================================================
# SUMMARY
# ============================================================================

echo
echo "==================================="
echo "Backup Complete!"
echo "==================================="
echo
echo "Location: $BACKUP_DIR"
echo
echo "Backed up:"
echo "  ✓ Current sysctl values (YOUR tuned settings)"
echo "  ✓ qBittorrent configuration"
echo "  ✓ Network device settings"
echo "  ✓ Existing tuning scripts"
echo "  ✓ Systemd services and overrides"
echo
echo "To restore your current settings later:"
echo "  sudo $BACKUP_DIR/RESTORE.sh"
echo
