#!/bin/bash
# install-racing-tuning-vm.sh
# VM-OPTIMIZED INSTALLER with SMART BACKUP
# Automatically detects VM, backs up YOUR current tuned settings
# Only installs VM-compatible tuning

set -e

echo "==================================="
echo "Seedbox Racing Tuning Installer"
echo "(VM Optimized with Smart Backup)"
echo "==================================="
echo

# ============================================================================
# CHECK IF RUNNING AS ROOT
# ============================================================================

if [ "$EUID" -ne 0 ]; then
    echo "✗ This script must be run as root or with sudo"
    exit 1
fi

# ============================================================================
# DETECT VM ENVIRONMENT
# ============================================================================

echo "Detecting environment..."

IS_VM=false

# Check for VM indicators
if [ -d "/sys/devices/virtual" ]; then
    IS_VM=true
fi

if systemd-detect-virt > /dev/null 2>&1; then
    VIRT_TYPE=$(systemd-detect-virt)
    if [ "$VIRT_TYPE" != "none" ]; then
        IS_VM=true
        echo "✓ VM detected: $VIRT_TYPE"
    fi
fi

# Check for physical CPU features
if [ ! -d "/sys/devices/system/cpu/cpu0/cpufreq" ]; then
    IS_VM=true
    echo "✓ VM indicators: No CPU frequency scaling"
fi

if [ "$IS_VM" = true ]; then
    echo "✓ Running in virtualized environment"
    echo "  (Will skip CPU governor, C-states, physical NVMe tuning)"
else
    echo "✓ Running on physical hardware"
fi

echo

# ============================================================================
# DETECT SWIZZIN
# ============================================================================

SWIZZIN_DETECTED=false
SWIZZIN_USER=""

if [ -f "/install/.swizzin.lock" ] || [ -f "/etc/swizzin/sources/globals" ]; then
    SWIZZIN_DETECTED=true
    SWIZZIN_USER=$(ls /home | grep -v "lost+found" | head -1)
    
    if [ -z "$SWIZZIN_USER" ]; then
        echo "✗ Swizzin detected but could not find user"
        exit 1
    fi
    
    echo "✓ Swizzin installation detected"
    echo "✓ Swizzin user: $SWIZZIN_USER"
else
    echo "✓ Standard installation (non-Swizzin)"
fi

echo

# ============================================================================
# CHECK PREREQUISITES
# ============================================================================

echo "Checking prerequisites..."

REQUIRED_FILES=(
    "99-seedbox-racing.conf"
    "qBittorrent.conf"
)

# Add VM-specific file
if [ "$IS_VM" = true ]; then
    REQUIRED_FILES+=("tune-hardware-vm.sh")
    REQUIRED_FILES+=("seedbox-tuning-vm.service")
else
    REQUIRED_FILES+=("tune-hardware.sh")
    REQUIRED_FILES+=("tune-cpu-io.sh")
    REQUIRED_FILES+=("seedbox-tuning.service")
fi

for file in "${REQUIRED_FILES[@]}"; do
    if [ ! -f "$file" ]; then
        echo "✗ Missing required file: $file"
        exit 1
    fi
done

# Check for backup script
if [ ! -f "backup-current-config.sh" ]; then
    echo "✗ Missing backup-current-config.sh"
    exit 1
fi

echo "✓ All required files present"
echo

# ============================================================================
# EXPLAIN WHAT WILL BE DONE
# ============================================================================

echo "This installer will:"
echo "  1. SMART BACKUP your current settings (preserves YOUR tuning)"
echo "  2. Fix buffer bloat (786MB → 128MB)"
echo "  3. Enable TCP Fast Open"
echo "  4. Apply qBittorrent optimizations"
echo "  5. Enable real-time process priority"

if [ "$IS_VM" = true ]; then
    echo "  6. Apply VM-compatible network tuning"
    echo
    echo "⚠️  VM-specific: Will SKIP (not available in VM):"
    echo "     • CPU governor control"
    echo "     • CPU C-state tuning"
    echo "     • Physical NVMe I/O scheduler"
else
    echo "  6. Set CPU to performance mode"
    echo "  7. Configure NVMe I/O scheduler"
    echo "  8. Apply hardware-level tuning"
fi

echo
echo "⚠️  WARNING: This prioritizes performance over stability"
echo

read -p "Continue with installation? (y/N): " -n 1 -r
echo
if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "Installation cancelled"
    exit 0
fi

echo

# ============================================================================
# SMART BACKUP - Preserve YOUR current tuned settings
# ============================================================================

echo "==================================="
echo "STEP 1: Smart Backup"
echo "==================================="
echo

if [ -f "backup-current-config.sh" ]; then
    chmod +x backup-current-config.sh
    ./backup-current-config.sh
    
    # Find the backup directory that was just created
    BACKUP_DIR=$(ls -td /root/racing-tuning-backup-* 2>/dev/null | head -1)
    
    if [ -n "$BACKUP_DIR" ]; then
        echo
        echo "✓ Smart backup complete: $BACKUP_DIR"
        echo "  (Your current tuned settings are preserved)"
    else
        echo "⚠️  Backup may have failed, but continuing..."
    fi
else
    echo "✗ backup-current-config.sh not found"
    exit 1
fi

echo

# ============================================================================
# INSTALL SYSCTL CONFIGURATION
# ============================================================================

echo "==================================="
echo "STEP 2: System Tuning"
echo "==================================="
echo

echo "Installing sysctl configuration..."

cp 99-seedbox-racing.conf /etc/sysctl.d/99-seedbox-racing.conf
chmod 644 /etc/sysctl.d/99-seedbox-racing.conf
echo "✓ Installed sysctl configuration"

# Apply sysctl settings
sysctl -p /etc/sysctl.d/99-seedbox-racing.conf > /dev/null 2>&1 || \
    echo "⚠️  Some sysctl settings may not have applied"
echo "✓ Applied sysctl settings"

echo

# ============================================================================
# INSTALL HARDWARE TUNING
# ============================================================================

echo "Installing hardware tuning scripts..."

if [ "$IS_VM" = true ]; then
    # VM-optimized version only
    cp tune-hardware-vm.sh /usr/local/bin/tune-hardware-vm.sh
    chmod 755 /usr/local/bin/tune-hardware-vm.sh
    echo "✓ Installed VM-optimized hardware tuning script"
    
    # Install VM-optimized service
    cp seedbox-tuning-vm.service /etc/systemd/system/seedbox-tuning.service
    chmod 644 /etc/systemd/system/seedbox-tuning.service
    echo "✓ Installed systemd service"
    
    # Run tuning immediately
    /usr/local/bin/tune-hardware-vm.sh || echo "⚠️  Some hardware tuning had errors (may be normal in VM)"
else
    # Physical hardware version
    cp tune-hardware.sh /usr/local/bin/tune-hardware.sh
    cp tune-cpu-io.sh /usr/local/bin/tune-cpu-io.sh
    chmod 755 /usr/local/bin/tune-hardware.sh
    chmod 755 /usr/local/bin/tune-cpu-io.sh
    echo "✓ Installed hardware tuning scripts"
    
    cp seedbox-tuning.service /etc/systemd/system/seedbox-tuning.service
    chmod 644 /etc/systemd/system/seedbox-tuning.service
    echo "✓ Installed systemd service"
    
    # Run tuning immediately
    /usr/local/bin/tune-hardware.sh || echo "⚠️  Some hardware tuning had errors"
    /usr/local/bin/tune-cpu-io.sh || echo "⚠️  Some CPU/IO tuning had errors"
fi

# Enable systemd service
systemctl daemon-reload
systemctl enable seedbox-tuning.service > /dev/null 2>&1
echo "✓ Enabled boot-time tuning service"

echo

# ============================================================================
# INSTALL QBITTORRENT CONFIGURATION
# ============================================================================

echo "==================================="
echo "STEP 3: qBittorrent Configuration"
echo "==================================="
echo

if [ "$SWIZZIN_DETECTED" = true ]; then
    echo "Configuring qBittorrent (Swizzin mode)..."
    
    # Stop qBittorrent
    systemctl stop "qbittorrent@${SWIZZIN_USER}" 2>/dev/null || true
    echo "✓ Stopped qBittorrent"
    
    # Install configuration
    QBIT_CONFIG_DIR="/home/$SWIZZIN_USER/.config/qBittorrent"
    mkdir -p "$QBIT_CONFIG_DIR"
    
    # Replace username in config
    sed "s|/home/kevin|/home/${SWIZZIN_USER}|g" qBittorrent.conf > "$QBIT_CONFIG_DIR/qBittorrent.conf"
    chown -R "$SWIZZIN_USER:$SWIZZIN_USER" "$QBIT_CONFIG_DIR"
    echo "✓ Installed qBittorrent configuration"
    
    # Create systemd override for RT priority
    OVERRIDE_DIR="/etc/systemd/system/qbittorrent@.service.d"
    mkdir -p "$OVERRIDE_DIR"
    
    cat > "$OVERRIDE_DIR/racing-override.conf" <<'EOF'
[Service]
# Real-time scheduling for racing performance
CPUSchedulingPolicy=rr
CPUSchedulingPriority=20

# CPU affinity - pin to core 0
CPUAffinity=0

# I/O priority - real-time class
IOSchedulingClass=realtime
IOSchedulingPriority=2

# Resource limits
LimitNOFILE=1048576
LimitNPROC=16384
LimitMEMLOCK=8589934592

# OOM protection
OOMScoreAdjust=-500

# Security (relaxed for performance)
NoNewPrivileges=false
EOF
    
    echo "✓ Created systemd override for RT priority"
    
    # Add RT priority limits
    if ! grep -q "${SWIZZIN_USER}.*rtprio" /etc/security/limits.conf 2>/dev/null; then
        echo "${SWIZZIN_USER}  hard  rtprio  99" >> /etc/security/limits.conf
        echo "${SWIZZIN_USER}  soft  rtprio  99" >> /etc/security/limits.conf
        echo "✓ Added RT priority limits"
    else
        echo "✓ RT priority limits already present"
    fi
    
    # Reload and restart qBittorrent
    systemctl daemon-reload
    systemctl start "qbittorrent@${SWIZZIN_USER}"
    echo "✓ Restarted qBittorrent"
    
else
    echo "⚠️  Non-Swizzin installation detected"
    echo "    Please manually install qBittorrent configuration"
fi

echo

# ============================================================================
# VERIFICATION
# ============================================================================

echo "==================================="
echo "STEP 4: Verification"
echo "==================================="
echo

# Check sysctl
RMEM_MAX=$(sysctl -n net.core.rmem_max)
if [ "$RMEM_MAX" = "134217728" ]; then
    echo "✓ TCP buffers: Fixed (128MB)"
else
    echo "✗ TCP buffers: $RMEM_MAX (expected 134217728)"
fi

# Check TCP Fast Open
TFO=$(sysctl -n net.ipv4.tcp_fastopen)
if [ "$TFO" = "3" ]; then
    echo "✓ TCP Fast Open: Enabled"
else
    echo "✗ TCP Fast Open: $TFO (expected 3)"
fi

# Check qBittorrent (Swizzin only)
if [ "$SWIZZIN_DETECTED" = true ]; then
    sleep 2  # Give qBittorrent time to start
    
    if systemctl is-active --quiet "qbittorrent@${SWIZZIN_USER}"; then
        echo "✓ qBittorrent: Running"
        
        # Check RT priority
        QBIT_PID=$(pgrep -u "$SWIZZIN_USER" qbittorrent-nox | head -1)
        if [ -n "$QBIT_PID" ]; then
            POLICY=$(chrt -p "$QBIT_PID" 2>/dev/null | grep "scheduling policy" | awk '{print $3}' || echo "UNKNOWN")
            if [ "$POLICY" = "SCHED_RR" ]; then
                echo "✓ qBittorrent RT Priority: Enabled (SCHED_RR)"
            else
                echo "⚠️  qBittorrent RT Priority: $POLICY (expected SCHED_RR)"
            fi
        fi
    else
        echo "✗ qBittorrent: Not running"
    fi
fi

echo

# ============================================================================
# FINAL SUMMARY
# ============================================================================

echo "==================================="
echo "Installation Complete!"
echo "==================================="
echo
echo "📊 EXPECTED IMPROVEMENTS:"
echo "  • 500-600ms faster to first piece"
echo "  • Full 10Gbps bandwidth utilization"
echo "  • Lower latency jitter"
echo
echo "💾 BACKUP LOCATION:"
echo "  • Your settings backed up to: $BACKUP_DIR"
echo "  • To restore: sudo $BACKUP_DIR/RESTORE.sh"
echo
echo "🔄 RESTORE SCRIPT:"
echo "  • Full restore: sudo ./restore-original-config.sh"
echo "  • (Restores YOUR tuned settings, not defaults)"
echo

if [ "$IS_VM" = true ]; then
    echo "⚠️  VM ENVIRONMENT:"
    echo "  • Skipped: CPU governor, C-states, physical NVMe"
    echo "  • Applied: Network tuning, sysctl, qBittorrent config"
    echo
fi

echo "⚠️  NEXT STEPS:"
echo "  1. REBOOT for full effect: sudo reboot"
echo "  2. Monitor CPU/RAM usage after reboot"
echo "  3. Test racing performance"
echo
echo "🔍 VERIFICATION COMMANDS:"
echo "  • Check buffers: sysctl net.core.rmem_max"
echo "  • Check qBittorrent: systemctl status qbittorrent@${SWIZZIN_USER}"
echo "  • Check RT priority: chrt -p \$(pgrep qbittorrent)"
echo

read -p "Reboot now? (y/N): " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo "Rebooting in 3 seconds..."
    sleep 3
    reboot
else
    echo
    echo "Remember to reboot later: sudo reboot"
    echo "Good luck with your races! 🏆"
fi
