#!/bin/bash
# restore-original-config-vm.sh
# SMART RESTORE: Restores YOUR backed-up tuned settings (not system defaults)
# Works with the smart backup that preserves your existing tuning

set -e

echo "==================================="
echo "Racing Tuning Restore Script"
echo "(Smart Restore - Your Settings)"
echo "==================================="
echo

# Must run as root
if [ "$EUID" -ne 0 ]; then
    echo "✗ This script must be run as root or with sudo"
    exit 1
fi

# ============================================================================
# DETECT SWIZZIN
# ============================================================================

SWIZZIN_DETECTED=false
SWIZZIN_USER=""

if [ -f "/install/.swizzin.lock" ] || [ -f "/etc/swizzin/sources/globals" ]; then
    SWIZZIN_DETECTED=true
    SWIZZIN_USER=$(ls /home | grep -v "lost+found" | head -1)
    echo "✓ Swizzin installation detected"
    echo "✓ Swizzin user: $SWIZZIN_USER"
else
    echo "✓ Standard installation (non-Swizzin)"
fi

echo

# ============================================================================
# FIND BACKUP DIRECTORY
# ============================================================================

echo "Looking for backup directory..."

BACKUP_DIR=""
LATEST_BACKUP=$(ls -td /root/racing-tuning-backup-* 2>/dev/null | head -1)

if [ -n "$LATEST_BACKUP" ]; then
    BACKUP_DIR="$LATEST_BACKUP"
    echo "✓ Found backup: $BACKUP_DIR"
    
    # Check if it has the smart backup files
    if [ -f "$BACKUP_DIR/sysctl-restore.conf" ]; then
        echo "✓ Smart backup detected (preserves your tuning)"
    elif [ -f "$BACKUP_DIR/sysctl-all-current.txt" ]; then
        echo "✓ Full sysctl dump available"
    else
        echo "⚠️  Backup found but may be incomplete"
    fi
else
    echo "✗ No backup directory found"
    echo
    echo "Cannot restore without backup. Options:"
    echo "  1. System settings will revert to defaults (may lose your tuning)"
    echo "  2. Cancel and investigate"
    echo
    read -p "Continue with default restore anyway? (y/N): " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo "Restore cancelled"
        exit 0
    fi
fi

echo

# ============================================================================
# CONFIRM RESTORE
# ============================================================================

echo "⚠️  This will:"
echo "  • Remove all racing tuning configurations"
if [ -n "$BACKUP_DIR" ] && [ -f "$BACKUP_DIR/sysctl-restore.conf" ]; then
    echo "  • Restore YOUR previous tuned settings (from backup)"
else
    echo "  • Reset to system defaults (your tuning may be lost)"
fi
echo "  • Reset qBittorrent to previous state"
echo "  • Remove RT priority override"
echo "  • Require a reboot to take full effect"
echo

read -p "Continue with restore? (y/N): " -n 1 -r
echo
if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "Restore cancelled"
    exit 0
fi

echo

# ============================================================================
# STOP SERVICES
# ============================================================================

echo "Stopping services..."

# Stop qBittorrent
if [ "$SWIZZIN_DETECTED" = true ]; then
    systemctl stop "qbittorrent@${SWIZZIN_USER}" 2>/dev/null || true
    echo "✓ Stopped qBittorrent (Swizzin)"
else
    systemctl stop qbittorrent-racing.service 2>/dev/null || true
    systemctl stop qbittorrent.service 2>/dev/null || true
    echo "✓ Stopped qBittorrent"
fi

# Stop tuning service
systemctl stop seedbox-tuning.service 2>/dev/null || true

echo

# ============================================================================
# REMOVE RACING TUNING FILES
# ============================================================================

echo "Removing racing tuning files..."

# Remove sysctl config
if [ -f /etc/sysctl.d/99-seedbox-racing.conf ]; then
    rm /etc/sysctl.d/99-seedbox-racing.conf
    echo "✓ Removed sysctl racing config"
fi

# Remove hardware tuning scripts
if [ -f /usr/local/bin/tune-hardware.sh ]; then
    rm /usr/local/bin/tune-hardware.sh
    echo "✓ Removed tune-hardware.sh"
fi

if [ -f /usr/local/bin/tune-hardware-vm.sh ]; then
    rm /usr/local/bin/tune-hardware-vm.sh
    echo "✓ Removed tune-hardware-vm.sh"
fi

if [ -f /usr/local/bin/tune-cpu-io.sh ]; then
    rm /usr/local/bin/tune-cpu-io.sh
    echo "✓ Removed tune-cpu-io.sh"
fi

if [ -f /usr/local/bin/start-qbittorrent-racing.sh ]; then
    rm /usr/local/bin/start-qbittorrent-racing.sh
    echo "✓ Removed start-qbittorrent-racing.sh"
fi

# Remove systemd services
if [ -f /etc/systemd/system/seedbox-tuning.service ]; then
    systemctl disable seedbox-tuning.service 2>/dev/null || true
    rm /etc/systemd/system/seedbox-tuning.service
    echo "✓ Removed seedbox-tuning.service"
fi

if [ -f /etc/systemd/system/qbittorrent-racing.service ]; then
    systemctl disable qbittorrent-racing.service 2>/dev/null || true
    rm /etc/systemd/system/qbittorrent-racing.service
    echo "✓ Removed qbittorrent-racing.service"
fi

# Remove Swizzin override
if [ "$SWIZZIN_DETECTED" = true ]; then
    if [ -f /etc/systemd/system/qbittorrent@.service.d/racing-override.conf ]; then
        rm /etc/systemd/system/qbittorrent@.service.d/racing-override.conf
        rmdir /etc/systemd/system/qbittorrent@.service.d 2>/dev/null || true
        echo "✓ Removed Swizzin RT priority override"
    fi
fi

systemctl daemon-reload
echo "✓ Reloaded systemd"

echo

# ============================================================================
# RESTORE CONFIGURATIONS FROM SMART BACKUP
# ============================================================================

if [ -n "$BACKUP_DIR" ]; then
    echo "Restoring from smart backup..."
    
    # Restore sysctl settings (YOUR tuned values)
    if [ -f "$BACKUP_DIR/sysctl-restore.conf" ]; then
        echo "Restoring your sysctl settings..."
        sysctl -p "$BACKUP_DIR/sysctl-restore.conf" > /dev/null 2>&1 || \
            echo "⚠️  Some sysctl settings could not be restored"
        echo "✓ Restored your previous sysctl tuning"
    else
        echo "⚠️  sysctl-restore.conf not found in backup"
    fi
    
    # Restore original sysctl files if they exist
    if [ -f "$BACKUP_DIR/sysctl.conf.original" ]; then
        cp "$BACKUP_DIR/sysctl.conf.original" /etc/sysctl.conf
        echo "✓ Restored /etc/sysctl.conf"
    fi
    
    if [ -d "$BACKUP_DIR/sysctl.d" ]; then
        # Don't overwrite, just ensure racing config is gone
        echo "✓ Original /etc/sysctl.d preserved"
    fi
    
    # Restore qBittorrent config
    if [ "$SWIZZIN_DETECTED" = true ]; then
        if [ -f "$BACKUP_DIR/qBittorrent.conf.original" ]; then
            cp "$BACKUP_DIR/qBittorrent.conf.original" \
               "/home/$SWIZZIN_USER/.config/qBittorrent/qBittorrent.conf"
            chown "$SWIZZIN_USER:$SWIZZIN_USER" \
               "/home/$SWIZZIN_USER/.config/qBittorrent/qBittorrent.conf"
            echo "✓ Restored original qBittorrent.conf"
        else
            echo "⚠️  Original qBittorrent.conf not found in backup"
        fi
    fi
    
    # Restore limits.conf if it was modified
    if [ -f "$BACKUP_DIR/limits.conf.original" ]; then
        # Remove RT priority lines
        if [ "$SWIZZIN_DETECTED" = true ]; then
            sed -i "/^${SWIZZIN_USER}.*rtprio/d" /etc/security/limits.conf 2>/dev/null || true
            echo "✓ Removed RT priority limits"
        fi
    fi
    
    echo
else
    echo "⚠️  No backup found - cannot restore your specific settings"
    echo "    System will use current/default values"
    echo
fi

# ============================================================================
# RESTART SERVICES
# ============================================================================

echo "Restarting services..."

if [ "$SWIZZIN_DETECTED" = true ]; then
    systemctl start "qbittorrent@${SWIZZIN_USER}"
    echo "✓ Started qBittorrent (Swizzin)"
else
    echo "⚠️  Please manually start qBittorrent if needed"
fi

echo

# ============================================================================
# SUMMARY
# ============================================================================

echo "==================================="
echo "Restore Complete!"
echo "==================================="
echo
echo "✓ Racing tuning configurations removed"

if [ -n "$BACKUP_DIR" ] && [ -f "$BACKUP_DIR/sysctl-restore.conf" ]; then
    echo "✓ YOUR previous tuned settings restored"
    echo "  (Not system defaults - your tuning preserved)"
else
    echo "⚠️  Settings restored to current/default state"
fi

echo "✓ Services restarted"
echo
echo "📁 BACKUP PRESERVED:"
echo "  • Location: $BACKUP_DIR"
echo "  • Can re-apply racing tuning later if needed"
echo
echo "⚠️  IMPORTANT:"
echo "  • REBOOT REQUIRED for full effect: sudo reboot"
echo "  • Your network/kernel tuning preserved (not lost)"
echo
echo "📊 WHAT WAS RESTORED:"
if [ -n "$BACKUP_DIR" ] && [ -f "$BACKUP_DIR/sysctl-restore.conf" ]; then
    echo "  • TCP buffers: Back to your previous tuned values"
    echo "  • BBR/FQ: Preserved (was in your backup)"
    echo "  • Other kernel params: Restored to your settings"
else
    echo "  • Settings: Current/default values"
fi
echo "  • qBittorrent: Back to original config"
echo "  • Real-time priority: Removed"
echo "  • Racing-specific files: Removed"
echo
echo "🔍 VERIFY RESTORATION:"
echo "  • Check your sysctl: cat $BACKUP_DIR/sysctl-restore.conf"
echo "  • Current buffers: sysctl net.core.rmem_max"

if [ "$SWIZZIN_DETECTED" = true ]; then
    echo "  • qBittorrent: systemctl status qbittorrent@${SWIZZIN_USER}"
fi

echo

read -p "Reboot now? (y/N): " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    echo "Rebooting in 3 seconds..."
    sleep 3
    reboot
else
    echo "Remember to reboot later: sudo reboot"
    echo
    echo "Your previous tuning has been preserved!"
fi
