#!/bin/bash
# /usr/local/bin/tune-hardware-vm.sh
# VM-OPTIMIZED HARDWARE TUNING FOR SEEDBOX RACING
# Skips VM-incompatible features (CPU governor, C-states, physical NVMe)

set -e

echo "=== Seedbox Racing Hardware Tuning (VM Optimized) ==="

# ============================================================================
# DETECT NETWORK INTERFACE
# ============================================================================

# Auto-detect primary network interface
IFACE=$(ip route | grep default | awk '{print $5}' | head -1)

if [ -z "$IFACE" ]; then
    echo "✗ Could not detect network interface"
    exit 1
fi

echo "Network interface: $IFACE"

# Detect gateway
GATEWAY=$(ip route | grep default | awk '{print $3}' | head -1)
echo "Gateway: $GATEWAY"

# ============================================================================
# NETWORK INTERFACE TUNING
# ============================================================================

echo
echo "Tuning network interface..."

# Increase TX queue length
/sbin/ip link set dev $IFACE txqueuelen 10000
echo "✓ TX queue length: 10000"

# Set initial congestion window and receive window
if [ -n "$GATEWAY" ]; then
    ip route change default via $GATEWAY dev $IFACE initcwnd 150 initrwnd 150 2>/dev/null || \
    echo "⚠️  Could not set initcwnd/initrwnd (may not be supported in VM)"
fi

# ============================================================================
# ETHTOOL: INTERRUPT COALESCING - ULTRA LOW LATENCY
# ============================================================================

echo
echo "Configuring interrupt coalescing (if supported)..."

# Disable adaptive coalescing
ethtool -C $IFACE adaptive-rx off adaptive-tx off 2>/dev/null && \
    echo "✓ Disabled adaptive coalescing" || \
    echo "⚠️  Adaptive coalescing: Not supported (VM limitation)"

# Zero delay coalescing
ethtool -C $IFACE rx-usecs 0 tx-usecs 0 2>/dev/null && \
    echo "✓ Zero delay coalescing" || \
    echo "⚠️  Delay coalescing: Not supported (VM limitation)"

# Interrupt per packet
ethtool -C $IFACE rx-frames 1 tx-frames 1 2>/dev/null && \
    echo "✓ Interrupt per packet" || \
    echo "⚠️  Frame interrupt: Not supported (VM limitation)"

# ============================================================================
# ETHTOOL: OFFLOADS - REDUCE LATENCY (VM usually doesn't support)
# ============================================================================

echo
echo "Configuring hardware offloads (if supported)..."

# Try to disable batching offloads
ethtool -K $IFACE gro off gso off tso off 2>/dev/null && \
    echo "✓ Disabled GRO/GSO/TSO" || \
    echo "⚠️  Offload control: Not supported (VM limitation - virtio handles this)"

# Additional offloads
ethtool -K $IFACE sg off 2>/dev/null || true
ethtool -K $IFACE ufo off 2>/dev/null || true
ethtool -K $IFACE lro off 2>/dev/null || true

# ============================================================================
# RING BUFFER TUNING (VM usually doesn't support)
# ============================================================================

echo
echo "Configuring ring buffers (if supported)..."

# Get current max values
RX_MAX=$(ethtool -g $IFACE 2>/dev/null | grep -A4 "Pre-set" | grep "RX:" | awk '{print $2}')
TX_MAX=$(ethtool -g $IFACE 2>/dev/null | grep -A4 "Pre-set" | grep "TX:" | awk '{print $2}')

if [ -n "$RX_MAX" ] && [ -n "$TX_MAX" ]; then
    ethtool -G $IFACE rx $RX_MAX tx $TX_MAX 2>/dev/null && \
        echo "✓ Ring buffers: RX=$RX_MAX, TX=$TX_MAX" || \
        echo "⚠️  Ring buffers: Not configurable (VM limitation)"
else
    echo "⚠️  Ring buffers: Not available (VM limitation)"
fi

# ============================================================================
# IRQ AFFINITY - PIN NETWORK INTERRUPTS (works in some VMs)
# ============================================================================

echo
echo "Configuring IRQ affinity..."

# Find IRQ numbers for the network interface
IRQS=$(grep $IFACE /proc/interrupts | awk -F: '{print $1}' | tr -d ' ')

if [ -z "$IRQS" ]; then
    echo "⚠️  IRQ affinity: Not available (VM limitation or virtio)"
else
    # With 2 cores: Pin to core 1 (leave core 0 for qBittorrent)
    CORE0_MASK="01"
    CORE1_MASK="02"
    
    IRQ_COUNT=$(echo "$IRQS" | wc -l)
    CORE_COUNTER=0
    
    for IRQ in $IRQS; do
        if [ -f /proc/irq/$IRQ/smp_affinity ]; then
            if [ "$IRQ_COUNT" -eq 1 ]; then
                echo $CORE1_MASK > /proc/irq/$IRQ/smp_affinity 2>/dev/null && \
                    echo "✓ IRQ $IRQ -> CPU 1" || \
                    echo "⚠️  IRQ $IRQ affinity failed"
            else
                if [ $((CORE_COUNTER % 2)) -eq 0 ]; then
                    echo $CORE0_MASK > /proc/irq/$IRQ/smp_affinity 2>/dev/null || true
                    echo "✓ IRQ $IRQ -> CPU 0"
                else
                    echo $CORE1_MASK > /proc/irq/$IRQ/smp_affinity 2>/dev/null || true
                    echo "✓ IRQ $IRQ -> CPU 1"
                fi
                CORE_COUNTER=$((CORE_COUNTER + 1))
            fi
        fi
    done
fi

# ============================================================================
# RPS/RFS TUNING - RECEIVE PACKET STEERING (works in VMs)
# ============================================================================

echo
echo "Configuring RPS/RFS for multicore packet processing..."

# Enable RPS on both cores (mask 03 = use CPU 0 and 1)
RPS_CONFIGURED=false
for rx_queue in /sys/class/net/$IFACE/queues/rx-*/rps_cpus; do
    if [ -f "$rx_queue" ]; then
        echo "03" > $rx_queue 2>/dev/null && RPS_CONFIGURED=true
    fi
done

if [ "$RPS_CONFIGURED" = true ]; then
    echo "✓ RPS enabled on both cores"
else
    echo "⚠️  RPS: Not available (single queue or VM limitation)"
fi

# Configure RFS
for rx_queue in /sys/class/net/$IFACE/queues/rx-*/rps_flow_cnt; do
    if [ -f "$rx_queue" ]; then
        echo 16384 > $rx_queue 2>/dev/null
    fi
done

# ============================================================================
# XPS TUNING - TRANSMIT PACKET STEERING (works in VMs)
# ============================================================================

echo
echo "Configuring XPS for multicore packet transmission..."

TX_QUEUES=$(ls -d /sys/class/net/$IFACE/queues/tx-* 2>/dev/null | wc -l)

if [ "$TX_QUEUES" -gt 0 ]; then
    QUEUE_NUM=0
    for xps_queue in /sys/class/net/$IFACE/queues/tx-*/xps_cpus; do
        if [ -f "$xps_queue" ]; then
            if [ $((QUEUE_NUM % 2)) -eq 0 ]; then
                echo "01" > $xps_queue 2>/dev/null || true
            else
                echo "02" > $xps_queue 2>/dev/null || true
            fi
            QUEUE_NUM=$((QUEUE_NUM + 1))
        fi
    done
    echo "✓ XPS configured"
else
    echo "⚠️  XPS: Not available (VM limitation)"
fi

# ============================================================================
# PAUSE FRAMES (works in some VMs)
# ============================================================================

echo
echo "Configuring flow control..."

ethtool -A $IFACE rx off tx off 2>/dev/null && \
    echo "✓ Disabled pause frames" || \
    echo "⚠️  Pause frames: Not configurable (VM limitation)"

# ============================================================================
# MTU (works in VMs)
# ============================================================================

echo
echo "Checking MTU..."

CURRENT_MTU=$(cat /sys/class/net/$IFACE/mtu)
echo "✓ Current MTU: $CURRENT_MTU"

# Don't change MTU unless specifically needed
# ip link set dev $IFACE mtu 1500

echo
echo "=== VM-optimized hardware tuning complete ==="
echo
echo "⚠️  Note: Some features skipped (VM limitations):"
echo "  • CPU governor control (no cpufreq in VM)"
echo "  • CPU C-states (no cpuidle in VM)"
echo "  • Physical NVMe tuning (virtual storage)"
echo "  • Some ethtool features (virtio limitations)"
echo
echo "✓ Applied VM-compatible optimizations:"
echo "  • TX queue length increased"
echo "  • RPS/RFS configured"
echo "  • IRQ affinity (if available)"
echo "  • Network parameters optimized"
